'use strict';
function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
import randomstring from 'randomstring';
import SynchronizationListener from '../../../clients/metaApi/synchronizationListener';
// import {NotFoundError} from '../../../clients/errorHandler';
import LoggerManager from '../../../logger';
let PeriodStatisticsStreamManager = class PeriodStatisticsStreamManager {
    /**
   * Returns listeners for a tracker
   * @param {string} accountId account id to return listeners for
   * @param {string} trackerId tracker id to return listeners for
   * @returns {{[listenerId: string]: PeriodStatisticsListener}} dictionary of period statistics listeners
   */ getTrackerListeners(accountId, trackerId) {
        if (!this._periodStatisticsListeners[accountId] || !this._periodStatisticsListeners[accountId][trackerId]) {
            return {};
        } else {
            return this._periodStatisticsListeners[accountId][trackerId];
        }
    }
    /**
   * Adds a period statistics event listener
   * @param {PeriodStatisticsListener} listener period statistics event listener
   * @param {String} accountId account id
   * @param {String} trackerId tracker id
   * @returns {String} listener id
   */ // eslint-disable-next-line complexity, max-statements
    addPeriodStatisticsListener(listener, accountId, trackerId) {
        var _this = this;
        return _async_to_generator(function*() {
            let newTracker = false;
            if (!_this._periodStatisticsCaches[accountId]) {
                _this._periodStatisticsCaches[accountId] = {};
            }
            if (!_this._periodStatisticsCaches[accountId][trackerId]) {
                newTracker = true;
                _this._periodStatisticsCaches[accountId][trackerId] = {
                    trackerData: {},
                    record: {},
                    lastPeriod: {},
                    equityAdjustments: {}
                };
            }
            const cache = _this._periodStatisticsCaches[accountId][trackerId];
            let connection = null;
            let retryIntervalInSeconds = _this._retryIntervalInSeconds;
            const equityTrackingClient = _this._equityTrackingClient;
            const listenerId = randomstring.generate(10);
            const removePeriodStatisticsListener = _this.removePeriodStatisticsListener;
            const getTrackerListeners = ()=>_this.getTrackerListeners(accountId, trackerId);
            const pendingInitalizationResolves = _this._pendingInitalizationResolves;
            const synchronizationFlags = _this._accountSynchronizationFlags;
            let PeriodStatisticsStreamListener = class PeriodStatisticsStreamListener extends SynchronizationListener {
                onDealsSynchronized(instanceIndex, synchronizationId) {
                    var _this = this;
                    return _async_to_generator(function*() {
                        try {
                            if (!synchronizationFlags[accountId]) {
                                synchronizationFlags[accountId] = true;
                                Object.values(getTrackerListeners()).forEach((accountListener)=>{
                                    accountListener.onConnected();
                                });
                                if (pendingInitalizationResolves[accountId]) {
                                    pendingInitalizationResolves[accountId].forEach((resolve)=>resolve());
                                    delete pendingInitalizationResolves[accountId];
                                }
                            }
                        } catch (err) {
                            listener.onError(err);
                            _this._logger.error('Error processing onDealsSynchronized event for ' + `equity chart listener for account ${accountId}`, err);
                        }
                    })();
                }
                onDisconnected(instanceIndex) {
                    var _this = this;
                    return _async_to_generator(function*() {
                        try {
                            if (synchronizationFlags[accountId] && !connection.healthMonitor.healthStatus.synchronized) {
                                synchronizationFlags[accountId] = false;
                                Object.values(getTrackerListeners()).forEach((trackerListener)=>{
                                    trackerListener.onDisconnected();
                                });
                            }
                        } catch (err) {
                            Object.values(getTrackerListeners()).forEach((trackerListener)=>{
                                trackerListener.onError(err);
                            });
                            _this._logger.error('Error processing onDisconnected event for ' + `equity chart listener for account ${accountId}`, err);
                        }
                    })();
                }
                // eslint-disable-next-line complexity, max-statements
                onSymbolPriceUpdated(instanceIndex, price) {
                    var _this = this;
                    return _async_to_generator(function*() {
                        try {
                            if (pendingInitalizationResolves[accountId]) {
                                pendingInitalizationResolves[accountId].forEach((resolve)=>resolve());
                                delete pendingInitalizationResolves[accountId];
                            }
                            if (!cache.lastPeriod) {
                                return;
                            }
                            /**
           * Process brokerTime:
           * - smaller than tracker startBrokerTime -> ignore
           * - bigger than tracker endBrokerTime -> send onTrackerCompleted, close connection
           * - bigger than period endBrokerTime -> send onPeriodStatisticsCompleted
           * - normal -> compare to previous data, if different -> send onPeriodStatisticsUpdated
           */ const equity = price.equity - Object.values(cache.equityAdjustments).reduce((a, b)=>a + b, 0);
                            const brokerTime = price.brokerTime;
                            if (brokerTime > cache.lastPeriod.endBrokerTime) {
                                Object.values(getTrackerListeners()).forEach((trackerListener)=>{
                                    trackerListener.onPeriodStatisticsCompleted();
                                });
                                cache.equityAdjustments = {};
                                const startBrokerTime = cache.lastPeriod.startBrokerTime;
                                cache.lastPeriod = null;
                                // eslint-disable-next-line no-constant-condition
                                while(true){
                                    let periods = yield equityTrackingClient.getTrackingStatistics(accountId, trackerId, undefined, 2, true);
                                    if (periods[0].startBrokerTime === startBrokerTime) {
                                        yield new Promise((res)=>setTimeout(res, 10000));
                                    } else {
                                        cache.lastPeriod = periods[0];
                                        periods.reverse();
                                        Object.values(getTrackerListeners()).forEach((trackerListener)=>{
                                            trackerListener.onPeriodStatisticsUpdated(periods);
                                        });
                                        break;
                                    }
                                }
                            } else {
                                if (cache.trackerData.startBrokerTime && brokerTime < cache.trackerData.startBrokerTime) {
                                    return;
                                }
                                if (cache.trackerData.endBrokerTime && brokerTime > cache.trackerData.endBrokerTime) {
                                    Object.values(getTrackerListeners()).forEach((trackerListener)=>{
                                        trackerListener.onTrackerCompleted();
                                    });
                                    cache.equityAdjustments = {};
                                    Object.keys(getTrackerListeners()).forEach((trackerListenerId)=>{
                                        removePeriodStatisticsListener(trackerListenerId);
                                    });
                                }
                                let absoluteDrawdown = Math.max(0, cache.lastPeriod.initialBalance - equity);
                                let relativeDrawdown = absoluteDrawdown / cache.lastPeriod.initialBalance;
                                let absoluteProfit = Math.max(0, equity - cache.lastPeriod.initialBalance);
                                let relativeProfit = absoluteProfit / cache.lastPeriod.initialBalance;
                                const previousRecord = JSON.stringify(cache.record);
                                if (!cache.record.thresholdExceeded) {
                                    if (cache.record.maxAbsoluteDrawdown < absoluteDrawdown) {
                                        cache.record.maxAbsoluteDrawdown = absoluteDrawdown;
                                        cache.record.maxRelativeDrawdown = relativeDrawdown;
                                        cache.record.maxDrawdownTime = brokerTime;
                                        if (cache.trackerData.relativeDrawdownThreshold && cache.trackerData.relativeDrawdownThreshold < relativeDrawdown || cache.trackerData.absoluteDrawdownThreshold && cache.trackerData.absoluteDrawdownThreshold < absoluteDrawdown) {
                                            cache.record.thresholdExceeded = true;
                                            cache.record.exceededThresholdType = 'drawdown';
                                        }
                                    }
                                    if (cache.record.maxAbsoluteProfit < absoluteProfit) {
                                        cache.record.maxAbsoluteProfit = absoluteProfit;
                                        cache.record.maxRelativeProfit = relativeProfit;
                                        cache.record.maxProfitTime = brokerTime;
                                        if (cache.trackerData.relativeProfitThreshold && cache.trackerData.relativeProfitThreshold < relativeProfit || cache.trackerData.absoluteProfitThreshold && cache.trackerData.absoluteProfitThreshold < absoluteProfit) {
                                            cache.record.thresholdExceeded = true;
                                            cache.record.exceededThresholdType = 'profit';
                                        }
                                    }
                                    if (JSON.stringify(cache.record) !== previousRecord) {
                                        Object.values(getTrackerListeners()).forEach((trackerListener)=>{
                                            trackerListener.onPeriodStatisticsUpdated([
                                                {
                                                    startBrokerTime: cache.lastPeriod.startBrokerTime,
                                                    endBrokerTime: cache.lastPeriod.endBrokerTime,
                                                    initialBalance: cache.lastPeriod.initialBalance,
                                                    maxAbsoluteDrawdown: cache.record.maxAbsoluteDrawdown,
                                                    maxAbsoluteProfit: cache.record.maxAbsoluteProfit,
                                                    maxDrawdownTime: cache.record.maxDrawdownTime,
                                                    maxProfitTime: cache.record.maxProfitTime,
                                                    maxRelativeDrawdown: cache.record.maxRelativeDrawdown,
                                                    maxRelativeProfit: cache.record.maxRelativeProfit,
                                                    period: cache.lastPeriod.period,
                                                    exceededThresholdType: cache.record.exceededThresholdType,
                                                    thresholdExceeded: cache.record.thresholdExceeded,
                                                    tradeDayCount: cache.record.tradeDayCount
                                                }
                                            ]);
                                        });
                                    }
                                }
                            }
                        } catch (err) {
                            Object.values(getTrackerListeners()).forEach((trackerListener)=>{
                                trackerListener.onError(err);
                            });
                            _this._logger.error('Error processing onSymbolPriceUpdated event for ' + `period statistics listener for account ${accountId}`, err);
                        }
                    })();
                }
                onDealAdded(instanceIndex, deal) {
                    var _this = this;
                    return _async_to_generator(function*() {
                        try {
                            if (!cache.lastPeriod || !Object.keys(cache.lastPeriod).length) {
                                return;
                            }
                            if (deal.type === 'DEAL_TYPE_BALANCE') {
                                cache.equityAdjustments[deal.id] = deal.profit;
                            }
                            const ignoredDealTypes = [
                                'DEAL_TYPE_BALANCE',
                                'DEAL_TYPE_CREDIT'
                            ];
                            if (!ignoredDealTypes.includes(deal.type)) {
                                const timeDiff = new Date(deal.time).getTime() - new Date(deal.brokerTime).getTime();
                                const startSearchDate = new Date(new Date(cache.lastPeriod.startBrokerTime).getTime() + timeDiff);
                                const deals = connection.historyStorage.getDealsByTimeRange(startSearchDate, new Date(8640000000000000)).filter((dealItem)=>!ignoredDealTypes.includes(dealItem.type));
                                deals.push(deal);
                                const tradedDays = {};
                                deals.forEach((dealItem)=>{
                                    tradedDays[dealItem.brokerTime.slice(0, 10)] = true;
                                });
                                const tradeDayCount = Object.keys(tradedDays).length;
                                if (cache.record.tradeDayCount !== tradeDayCount) {
                                    cache.record.tradeDayCount = tradeDayCount;
                                    Object.values(getTrackerListeners()).forEach((trackerListener)=>{
                                        trackerListener.onPeriodStatisticsUpdated([
                                            {
                                                startBrokerTime: cache.lastPeriod.startBrokerTime,
                                                endBrokerTime: cache.lastPeriod.endBrokerTime,
                                                initialBalance: cache.lastPeriod.initialBalance,
                                                maxAbsoluteDrawdown: cache.record.maxAbsoluteDrawdown,
                                                maxAbsoluteProfit: cache.record.maxAbsoluteProfit,
                                                maxDrawdownTime: cache.record.maxDrawdownTime,
                                                maxProfitTime: cache.record.maxProfitTime,
                                                maxRelativeDrawdown: cache.record.maxRelativeDrawdown,
                                                maxRelativeProfit: cache.record.maxRelativeProfit,
                                                period: cache.lastPeriod.period,
                                                exceededThresholdType: cache.record.exceededThresholdType,
                                                thresholdExceeded: cache.record.thresholdExceeded,
                                                tradeDayCount: cache.record.tradeDayCount
                                            }
                                        ]);
                                    });
                                }
                            }
                        } catch (err) {
                            Object.values(getTrackerListeners()).forEach((trackerListener)=>{
                                trackerListener.onError(err);
                            });
                            _this._logger.error('Error processing onDealAdded event for ' + `period statistics listener for account ${accountId}`, err);
                        }
                    })();
                }
            };
            const account = yield _this._metaApi.metatraderAccountApi.getAccount(accountId);
            const tracker = yield equityTrackingClient.getTracker(accountId, trackerId);
            cache.trackerData = tracker;
            if (!_this._periodStatisticsListeners[accountId]) {
                _this._periodStatisticsListeners[accountId] = {};
            }
            if (!_this._periodStatisticsListeners[accountId][trackerId]) {
                _this._periodStatisticsListeners[accountId][trackerId] = {};
            }
            const accountListeners = _this._periodStatisticsListeners[accountId][trackerId];
            accountListeners[listenerId] = listener;
            _this._accountsByListenerId[listenerId] = accountId;
            _this._trackersByListenerId[listenerId] = trackerId;
            let isDeployed = false;
            while(!isDeployed){
                try {
                    yield account.waitDeployed();
                    isDeployed = true;
                } catch (err) {
                    listener.onError(err);
                    _this._logger.error(`Error wait for account ${accountId} to deploy, retrying`, err);
                    yield new Promise((res)=>setTimeout(res, retryIntervalInSeconds * 1000));
                    retryIntervalInSeconds = Math.min(retryIntervalInSeconds * 2, 300);
                }
            }
            if (!_this._periodStatisticsConnections[accountId]) {
                retryIntervalInSeconds = _this._retryIntervalInSeconds;
                connection = account.getStreamingConnection();
                const syncListener = new PeriodStatisticsStreamListener();
                connection.addSynchronizationListener(syncListener);
                _this._periodStatisticsConnections[accountId] = connection;
                _this._syncListeners[trackerId] = syncListener;
                let isSynchronized = false;
                while(!isSynchronized){
                    try {
                        yield connection.connect();
                        yield connection.waitSynchronized();
                        isSynchronized = true;
                    } catch (err) {
                        listener.onError(err);
                        _this._logger.error('Error configuring period statistics stream listener for ' + `account ${accountId}, retrying`, err);
                        yield new Promise((res)=>setTimeout(res, retryIntervalInSeconds * 1000));
                        retryIntervalInSeconds = Math.min(retryIntervalInSeconds * 2, 300);
                    }
                }
                retryIntervalInSeconds = _this._retryIntervalInSeconds;
            } else {
                connection = _this._periodStatisticsConnections[accountId];
                if (newTracker) {
                    const syncListener = new PeriodStatisticsStreamListener();
                    connection.addSynchronizationListener(syncListener);
                    _this._syncListeners[trackerId] = syncListener;
                }
                if (!connection.healthMonitor.healthStatus.synchronized) {
                    if (!_this._pendingInitalizationResolves[accountId]) {
                        _this._pendingInitalizationResolves[accountId] = [];
                    }
                    let resolveInitialize;
                    let initializePromise = new Promise((res, rej)=>{
                        resolveInitialize = res;
                    });
                    _this._pendingInitalizationResolves[accountId].push(resolveInitialize);
                    yield initializePromise;
                }
            }
            let initialData = [];
            const fetchInitialData = function() {
                var _ref = _async_to_generator(function*() {
                    try {
                        initialData = yield equityTrackingClient.getTrackingStatistics(accountId, trackerId, undefined, undefined, true);
                        if (initialData.length) {
                            const lastItem = initialData[0];
                            if (_this._fetchInitialDataIntervalId[listenerId]) {
                                clearInterval(_this._fetchInitialDataIntervalId[listenerId]);
                                delete _this._fetchInitialDataIntervalId[listenerId];
                            }
                            listener.onPeriodStatisticsUpdated(initialData);
                            cache.lastPeriod = {
                                startBrokerTime: lastItem.startBrokerTime,
                                endBrokerTime: lastItem.endBrokerTime,
                                period: lastItem.period,
                                initialBalance: lastItem.initialBalance,
                                maxDrawdownTime: lastItem.maxDrawdownTime,
                                maxAbsoluteDrawdown: lastItem.maxAbsoluteDrawdown,
                                maxRelativeDrawdown: lastItem.maxRelativeDrawdown,
                                maxProfitTime: lastItem.maxProfitTime,
                                maxAbsoluteProfit: lastItem.maxAbsoluteProfit,
                                maxRelativeProfit: lastItem.maxRelativeProfit,
                                thresholdExceeded: lastItem.thresholdExceeded,
                                exceededThresholdType: lastItem.exceededThresholdType,
                                tradeDayCount: lastItem.tradeDayCount
                            };
                            cache.record = cache.lastPeriod;
                        }
                    } catch (err) {
                        listener.onError(err);
                        _this._logger.error(`Failed to initialize tracking statistics data for account ${accountId}`, err);
                        yield new Promise((res)=>setTimeout(res, retryIntervalInSeconds * 1000));
                        retryIntervalInSeconds = Math.min(retryIntervalInSeconds * 2, 300);
                    }
                });
                return function fetchInitialData() {
                    return _ref.apply(this, arguments);
                };
            }();
            retryIntervalInSeconds = _this._retryIntervalInSeconds;
            _this._fetchInitialDataIntervalId[listenerId] = setInterval(fetchInitialData, retryIntervalInSeconds * 1000 * 2 * 60);
            fetchInitialData();
            return listenerId;
        })();
    }
    /**
   * Removes period statistics event listener by id
   * @param {String} listenerId listener id 
   */ // eslint-disable-next-line complexity
    removePeriodStatisticsListener(listenerId) {
        if (this._accountsByListenerId[listenerId] && this._trackersByListenerId[listenerId]) {
            if (this._fetchInitialDataIntervalId[listenerId]) {
                clearInterval(this._fetchInitialDataIntervalId[listenerId]);
                delete this._fetchInitialDataIntervalId[listenerId];
            }
            const accountId = this._accountsByListenerId[listenerId];
            const trackerId = this._trackersByListenerId[listenerId];
            delete this._accountsByListenerId[listenerId];
            delete this._trackersByListenerId[listenerId];
            if (this._periodStatisticsListeners[accountId]) {
                if (this._periodStatisticsListeners[accountId][trackerId]) {
                    delete this._periodStatisticsListeners[accountId][trackerId][listenerId];
                    if (!Object.keys(this._periodStatisticsListeners[accountId][trackerId]).length) {
                        delete this._periodStatisticsListeners[accountId][trackerId];
                        if (this._periodStatisticsConnections[accountId] && this._syncListeners[trackerId]) {
                            this._periodStatisticsConnections[accountId].removeSynchronizationListener(this._syncListeners[trackerId]);
                            delete this._syncListeners[trackerId];
                        }
                    }
                }
                if (!Object.keys(this._periodStatisticsListeners[accountId]).length) {
                    delete this._periodStatisticsListeners[accountId];
                }
            }
            if (this._periodStatisticsConnections[accountId] && !this._periodStatisticsListeners[accountId]) {
                delete this._accountSynchronizationFlags[accountId];
                this._periodStatisticsConnections[accountId].close();
                delete this._periodStatisticsConnections[accountId];
            }
        }
    }
    /**
   * Constructs period statistics event listener manager instance
   * @param {DomainClient} domainClient domain client
   * @param {EquityTrackingClient} equityTrackingClient equity tracking client
   * @param {MetaApi} metaApi metaApi SDK instance
   */ constructor(domainClient, equityTrackingClient, metaApi){
        this._domainClient = domainClient;
        this._equityTrackingClient = equityTrackingClient;
        this._metaApi = metaApi;
        this._periodStatisticsListeners = {};
        this._accountsByListenerId = {};
        this._trackersByListenerId = {};
        this._trackerSyncListeners = {};
        this._periodStatisticsConnections = {};
        this._periodStatisticsCaches = {};
        this._accountSynchronizationFlags = {};
        this._pendingInitalizationResolves = {};
        this._syncListeners = {};
        this._retryIntervalInSeconds = 1;
        this._fetchInitialDataIntervalId = {};
        this.removePeriodStatisticsListener = this.removePeriodStatisticsListener.bind(this);
        this._logger = LoggerManager.getLogger('PeriodStatisticsStreamManager');
    }
};
/**
 * Manager for handling period statistics event listeners
 */ export { PeriodStatisticsStreamManager as default };

//# sourceMappingURL=data:application/json;base64,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