'use strict';
function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
import TerminalState from './terminalState';
import MemoryHistoryStorage from './memoryHistoryStorage';
import TimeoutError from '../clients/timeoutError';
import randomstring from 'randomstring';
import ConnectionHealthMonitor from './connectionHealthMonitor';
import { ValidationError } from '../clients/errorHandler';
import OptionsValidator from '../clients/optionsValidator';
import LoggerManager from '../logger';
import MetaApiConnection from './metaApiConnection';
let StreamingMetaApiConnection = class StreamingMetaApiConnection extends MetaApiConnection {
    /**
   * Opens the connection. Can only be called the first time, next calls will be ignored.
   * @param {string} instanceId connection instance id
   * @return {Promise} promise resolving when the connection is opened
   */ connect(instanceId) {
        var _this = this;
        return _async_to_generator(function*() {
            if (!_this._openedInstances.includes(instanceId)) {
                _this._openedInstances.push(instanceId);
            }
            if (!_this._opened) {
                _this._logger.trace(`${_this._account.id}: Opening connection`);
                _this._opened = true;
                try {
                    yield _this.initialize();
                    yield _this.subscribe();
                } catch (err) {
                    yield _this.close();
                    throw err;
                }
            }
        })();
    }
    /**
   * Clears the order and transaction history of a specified application and removes application
   * @return {Promise} promise resolving when the history is cleared and application is removed
   */ removeApplication() {
        this._checkIsConnectionActive();
        this._historyStorage.clear();
        return this._websocketClient.removeApplication(this._account.id);
    }
    /**
   * Requests the terminal to start synchronization process
   * (see https://metaapi.cloud/docs/client/websocket/synchronizing/synchronize/)
   * @param {String} instanceIndex instance index
   * @returns {Promise} promise which resolves when synchronization started
   */ synchronize(instanceIndex) {
        var _this = this;
        return _async_to_generator(function*() {
            _this._checkIsConnectionActive();
            const region = _this.getRegion(instanceIndex);
            const instance = _this.getInstanceNumber(instanceIndex);
            const host = _this.getHostName(instanceIndex);
            let startingHistoryOrderTime = new Date(Math.max((_this._historyStartTime || new Date(0)).getTime(), (yield _this._historyStorage.lastHistoryOrderTime(instance)).getTime()));
            let startingDealTime = new Date(Math.max((_this._historyStartTime || new Date(0)).getTime(), (yield _this._historyStorage.lastDealTime(instance)).getTime()));
            let synchronizationId = randomstring.generate(32);
            _this._getState(instanceIndex).lastSynchronizationId = synchronizationId;
            const accountId = _this._account.accountRegions[region];
            _this._logger.debug(`${_this._account.id}:${instanceIndex}: initiating synchronization ${synchronizationId}`);
            return _this._websocketClient.synchronize(accountId, instance, host, synchronizationId, startingHistoryOrderTime, startingDealTime, _this.terminalState.getHashes());
        })();
    }
    /**
   * Initializes meta api connection
   * @return {Promise} promise which resolves when meta api connection is initialized
   */ initialize() {
        var _this = this;
        return _async_to_generator(function*() {
            _this._checkIsConnectionActive();
            yield _this._historyStorage.initialize(_this._account.id, _this._connectionRegistry.application);
            _this._websocketClient.addAccountCache(_this._account.id, _this._account.accountRegions);
        })();
    }
    /**
   * Initiates subscription to MetaTrader terminal
   * @returns {Promise} promise which resolves when subscription is initiated
   */ subscribe() {
        var _this = this;
        return _async_to_generator(function*() {
            _this._checkIsConnectionActive();
            const accountRegions = _this._account.accountRegions;
            Object.entries(accountRegions).forEach(([region, replicaId])=>{
                if (!_this._options.region || _this._options.region === region) {
                    _this._websocketClient.ensureSubscribe(replicaId, 0);
                    _this._websocketClient.ensureSubscribe(replicaId, 1);
                }
            });
        })();
    }
    /**
   * Subscribes on market data of specified symbol (see
   * https://metaapi.cloud/docs/client/websocket/marketDataStreaming/subscribeToMarketData/).
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @param {Array<MarketDataSubscription>} subscriptions array of market data subscription to create or update. Please
   * note that this feature is not fully implemented on server-side yet
   * @param {number} [timeoutInSeconds] timeout to wait for prices in seconds, default is 30
   * @param {boolean} [waitForQuote] if set to false, the method will resolve without waiting for the first quote to
   * arrive. Default is to wait for quote if quotes subscription is requested.
   * @returns {Promise} promise which resolves when subscription request was processed
   */ subscribeToMarketData(symbol, subscriptions, timeoutInSeconds, waitForQuote = true) {
        var _this = this;
        return _async_to_generator(function*() {
            _this._checkIsConnectionActive();
            if (!_this._terminalState.specification(symbol)) {
                throw new ValidationError(`${_this._account.id}: Cannot subscribe to market data for symbol ${symbol} because ` + 'symbol does not exist');
            } else {
                subscriptions = subscriptions || [
                    {
                        type: 'quotes'
                    }
                ];
                if (_this._subscriptions[symbol]) {
                    const prevSubscriptions = _this._subscriptions[symbol].subscriptions;
                    subscriptions.forEach((subscription)=>{
                        const index = subscription.type === 'candles' ? prevSubscriptions.findIndex((item)=>item.type === subscription.type && item.timeframe === subscription.timeframe) : prevSubscriptions.findIndex((item)=>item.type === subscription.type);
                        if (index === -1) {
                            prevSubscriptions.push(subscription);
                        } else {
                            prevSubscriptions[index] = subscription;
                        }
                    });
                } else {
                    _this._subscriptions[symbol] = {
                        subscriptions
                    };
                }
                yield _this._websocketClient.subscribeToMarketData(_this._account.id, symbol, subscriptions, _this._account.reliability);
                if (waitForQuote !== false && subscriptions.find((s)=>s.type === 'quotes')) {
                    return _this.terminalState.waitForPrice(symbol, timeoutInSeconds);
                }
            }
        })();
    }
    /**
   * Unsubscribes from market data of specified symbol (see
   * https://metaapi.cloud/docs/client/websocket/marketDataStreaming/unsubscribeFromMarketData/).
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @param {Array<MarketDataUnsubscription>} unsubscriptions array of subscriptions to cancel
   * @returns {Promise} promise which resolves when unsubscription request was processed
   */ unsubscribeFromMarketData(symbol, unsubscriptions) {
        this._checkIsConnectionActive();
        if (!unsubscriptions) {
            delete this._subscriptions[symbol];
        } else if (this._subscriptions[symbol]) {
            this._subscriptions[symbol].subscriptions = this._subscriptions[symbol].subscriptions.filter((subscription)=>{
                return !unsubscriptions.find((unsubscription)=>subscription.type === unsubscription.type && (!unsubscription.timeframe || subscription.timeframe === unsubscription.timeframe));
            });
            if (!this._subscriptions[symbol].subscriptions.length) {
                delete this._subscriptions[symbol];
            }
        }
        return this._websocketClient.unsubscribeFromMarketData(this._account.id, symbol, unsubscriptions, this._account.reliability);
    }
    /**
   * Invoked when subscription downgrade has occurred
   * @param {String} instanceIndex index of an account instance connected
   * @param {string} symbol symbol to update subscriptions for
   * @param {Array<MarketDataSubscription>} updates array of market data subscription to update
   * @param {Array<MarketDataUnsubscription>} unsubscriptions array of subscriptions to cancel
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ // eslint-disable-next-line complexity
    onSubscriptionDowngraded(instanceIndex, symbol, updates, unsubscriptions) {
        var _this = this;
        return _async_to_generator(function*() {
            if (unsubscriptions === null || unsubscriptions === void 0 ? void 0 : unsubscriptions.length) {
                _this.unsubscribeFromMarketData(symbol, unsubscriptions).catch((err)=>{
                    let method = err.name !== 'ValidationError' ? 'error' : 'trace';
                    _this._logger[method](`${_this._account.id}: failed do unsubscribe from market data on subscription downgraded`, err);
                });
            }
            if (updates === null || updates === void 0 ? void 0 : updates.length) {
                _this.subscribeToMarketData(symbol, updates).catch((err)=>{
                    _this._logger.error(`${_this._account.id}: failed do subscribe from market data on subscription downgraded`, err);
                });
            }
        })();
    }
    /**
   * Returns list of the symbols connection is subscribed to
   * @returns {Array<String>} list of the symbols connection is subscribed to
   */ get subscribedSymbols() {
        return Object.keys(this._subscriptions);
    }
    /**
   * Returns subscriptions for a symbol
   * @param {string} symbol symbol to retrieve subscriptions for
   * @returns {Array<MarketDataSubscription>} list of market data subscriptions for the symbol
   */ subscriptions(symbol) {
        this._checkIsConnectionActive();
        return (this._subscriptions[symbol] || {}).subscriptions;
    }
    /**
   * Returns local copy of terminal state
   * @returns {TerminalState} local copy of terminal state
   */ get terminalState() {
        return this._terminalState;
    }
    /**
   * Returns local history storage
   * @returns {HistoryStorage} local history storage
   */ get historyStorage() {
        return this._historyStorage;
    }
    /**
   * Invoked when connection to MetaTrader terminal established
   * @param {String} instanceIndex index of an account instance connected
   * @param {Number} replicas number of account replicas launched
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onConnected(instanceIndex, replicas) {
        var _this = this;
        return _async_to_generator(function*() {
            let key = randomstring.generate(32);
            let state = _this._getState(instanceIndex);
            const region = _this.getRegion(instanceIndex);
            _this.cancelRefresh(region);
            yield _this._terminalHashManager.refreshIgnoredFieldLists(region);
            state.shouldSynchronize = key;
            state.synchronizationRetryIntervalInSeconds = 1;
            state.synchronized = false;
            _this._ensureSynchronized(instanceIndex, key);
            _this._logger.debug(`${_this._account.id}:${instanceIndex}: connected to broker`);
        })();
    }
    /**
   * Invoked when connection to MetaTrader terminal terminated
   * @param {String} instanceIndex index of an account instance connected
   */ onDisconnected(instanceIndex) {
        var _this = this;
        return _async_to_generator(function*() {
            let state = _this._getState(instanceIndex);
            state.lastDisconnectedSynchronizationId = state.lastSynchronizationId;
            state.lastSynchronizationId = undefined;
            state.shouldSynchronize = undefined;
            state.synchronized = false;
            state.disconnected = true;
            const instanceNumber = _this.getInstanceNumber(instanceIndex);
            const region = _this.getRegion(instanceIndex);
            const instance = `${region}:${instanceNumber}`;
            delete _this._refreshMarketDataSubscriptionSessions[instance];
            clearTimeout(_this._refreshMarketDataSubscriptionTimeouts[instance]);
            delete _this._refreshMarketDataSubscriptionTimeouts[instance];
            clearTimeout(state.synchronizationTimeout);
            delete state.synchronizationTimeout;
            clearTimeout(state.ensureSynchronizeTimeout);
            delete state.ensureSynchronizeTimeout;
            _this._logger.debug(`${_this._account.id}:${instanceIndex}: disconnected from broker`);
        })();
    }
    /**
   * Invoked when a symbol specifications were updated
   * @param {String} instanceIndex index of account instance connected
   * @param {Array<MetatraderSymbolSpecification>} specifications updated specifications
   * @param {Array<String>} removedSymbols removed symbols
   */ onSymbolSpecificationsUpdated(instanceIndex, specifications, removedSymbols) {
        this._scheduleSynchronizationTimeout(instanceIndex);
    }
    /**
   * Invoked when position synchronization finished to indicate progress of an initial terminal state synchronization
   * @param {string} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   */ onPositionsSynchronized(instanceIndex, synchronizationId) {
        this._scheduleSynchronizationTimeout(instanceIndex);
    }
    /**
   * Invoked when pending order synchronization fnished to indicate progress of an initial terminal state
   * synchronization
   * @param {string} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   */ onPendingOrdersSynchronized(instanceIndex, synchronizationId) {
        this._scheduleSynchronizationTimeout(instanceIndex);
    }
    /**
   * Invoked when a synchronization of history deals on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onDealsSynchronized(instanceIndex, synchronizationId) {
        var _this = this;
        return _async_to_generator(function*() {
            let state = _this._getState(instanceIndex);
            state.dealsSynchronized[synchronizationId] = true;
            _this._scheduleSynchronizationTimeout(instanceIndex);
            _this._logger.debug(`${_this._account.id}:${instanceIndex}: finished synchronization ${synchronizationId}`);
        })();
    }
    /**
   * Invoked when a synchronization of history orders on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onHistoryOrdersSynchronized(instanceIndex, synchronizationId) {
        var _this = this;
        return _async_to_generator(function*() {
            let state = _this._getState(instanceIndex);
            state.ordersSynchronized[synchronizationId] = true;
            _this._scheduleSynchronizationTimeout(instanceIndex);
        })();
    }
    /**
   * Invoked when connection to MetaApi websocket API restored after a disconnect
   * @param {String} region reconnected region
   * @param {Number} instanceNumber reconnected instance number
   * @return {Promise} promise which resolves when connection to MetaApi websocket API restored after a disconnect
   */ onReconnected(region, instanceNumber) {
        var _this = this;
        return _async_to_generator(function*() {
            const instanceTemplate = `${region}:${instanceNumber}`;
            Object.keys(_this._stateByInstanceIndex).filter((key)=>key.startsWith(`${instanceTemplate}:`)).forEach((key)=>{
                delete _this._stateByInstanceIndex[key];
            });
            delete _this._refreshMarketDataSubscriptionSessions[instanceTemplate];
            clearTimeout(_this._refreshMarketDataSubscriptionTimeouts[instanceTemplate]);
            delete _this._refreshMarketDataSubscriptionTimeouts[instanceTemplate];
        })();
    }
    /**
   * Invoked when a stream for an instance index is closed
   * @param {String} instanceIndex index of an account instance connected
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onStreamClosed(instanceIndex) {
        var _this = this;
        return _async_to_generator(function*() {
            delete _this._stateByInstanceIndex[instanceIndex];
        })();
    }
    /**
   * Invoked when MetaTrader terminal state synchronization is started
   * @param {string} instanceIndex index of an account instance connected
   * @param {string} specificationsHash specifications hash
   * @param {string} positionsHash positions hash
   * @param {string} ordersHash orders hash
   * @param {string} synchronizationId synchronization id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onSynchronizationStarted(instanceIndex, specificationsHash, positionsHash, ordersHash, synchronizationId) {
        var _this = this;
        return _async_to_generator(function*() {
            _this._logger.debug(`${_this._account.id}:${instanceIndex}: starting synchronization ${synchronizationId}`);
            const instanceNumber = _this.getInstanceNumber(instanceIndex);
            const region = _this.getRegion(instanceIndex);
            const instance = `${region}:${instanceNumber}`;
            const accountId = _this._account.accountRegions[region];
            delete _this._refreshMarketDataSubscriptionSessions[instance];
            let sessionId = randomstring.generate(32);
            _this._refreshMarketDataSubscriptionSessions[instance] = sessionId;
            clearTimeout(_this._refreshMarketDataSubscriptionTimeouts[instance]);
            delete _this._refreshMarketDataSubscriptionTimeouts[instance];
            yield _this._refreshMarketDataSubscriptions(accountId, instanceNumber, sessionId);
            _this._scheduleSynchronizationTimeout(instanceIndex);
            let state = _this._getState(instanceIndex);
            if (state && !_this._closed) {
                state.lastSynchronizationId = synchronizationId;
            }
        })();
    }
    /**
   * Invoked when account region has been unsubscribed
   * @param {String} region account region unsubscribed
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onUnsubscribeRegion(region) {
        Object.keys(this._refreshMarketDataSubscriptionTimeouts).filter((instance)=>instance.startsWith(`${region}:`)).forEach((instance)=>{
            clearTimeout(this._refreshMarketDataSubscriptionTimeouts[instance]);
            delete this._refreshMarketDataSubscriptionTimeouts[instance];
            delete this._refreshMarketDataSubscriptionSessions[instance];
        });
        Object.keys(this._stateByInstanceIndex).filter((instance)=>instance.startsWith(`${region}:`)).forEach((instance)=>delete this._stateByInstanceIndex[instance]);
    }
    /**
   * Returns flag indicating status of state synchronization with MetaTrader terminal
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId optional synchronization request id, last synchronization request id will be used
   * by default
   * @return {Promise<Boolean>} promise resolving with a flag indicating status of state synchronization with MetaTrader
   * terminal
   */ isSynchronized(instanceIndex, synchronizationId) {
        var _this = this;
        return _async_to_generator(function*() {
            return Object.values(_this._stateByInstanceIndex).reduce((acc, s)=>{
                if (instanceIndex !== undefined && s.instanceIndex !== instanceIndex) {
                    return acc;
                }
                const checkSynchronizationId = synchronizationId || s.lastSynchronizationId;
                let synchronized = !!s.ordersSynchronized[checkSynchronizationId] && !!s.dealsSynchronized[checkSynchronizationId];
                return acc || synchronized;
            }, false);
        })();
    }
    /**
   * @typedef {Object} SynchronizationOptions
   * @property {String} [applicationPattern] application regular expression pattern, default is .*
   * @property {String} [synchronizationId] synchronization id, last synchronization request id will be used by
   * default
   * @property {Number} [instanceIndex] index of an account instance to ensure synchronization on, default is to wait
   * for the first instance to synchronize
   * @property {Number} [timeoutInSeconds] wait timeout in seconds, default is 5m
   * @property {Number} [intervalInMilliseconds] interval between account reloads while waiting for a change, default is 1s
   */ /**
   * Waits until synchronization to MetaTrader terminal is completed
   * @param {SynchronizationOptions} opts synchronization options
   * @return {Promise} promise which resolves when synchronization to MetaTrader terminal is completed
   * @throws {TimeoutError} if application failed to synchronize with the teminal within timeout allowed
   */ // eslint-disable-next-line complexity
    waitSynchronized(opts) {
        var _this = this;
        return _async_to_generator(function*() {
            _this._checkIsConnectionActive();
            opts = opts || {};
            let instanceIndex = opts.instanceIndex;
            let synchronizationId = opts.synchronizationId;
            let timeoutInSeconds = opts.timeoutInSeconds || 300;
            let intervalInMilliseconds = opts.intervalInMilliseconds || 1000;
            let applicationPattern = opts.applicationPattern || (_this._account.application === 'CopyFactory' ? 'CopyFactory.*|RPC' : 'RPC');
            let startTime = Date.now();
            let synchronized;
            while(!(synchronized = yield _this.isSynchronized(instanceIndex, synchronizationId)) && startTime + timeoutInSeconds * 1000 > Date.now()){
                yield new Promise((res)=>setTimeout(res, intervalInMilliseconds));
            }
            let state;
            if (instanceIndex === undefined) {
                for (let s of Object.values(_this._stateByInstanceIndex)){
                    if (yield _this.isSynchronized(s.instanceIndex, synchronizationId)) {
                        state = s;
                        instanceIndex = s.instanceIndex;
                    }
                }
            } else {
                state = Object.values(_this._stateByInstanceIndex).find((s)=>s.instanceIndex === instanceIndex);
            }
            if (!synchronized) {
                throw new TimeoutError('Timed out waiting for MetaApi to synchronize to MetaTrader account ' + _this._account.id + ', synchronization id ' + (synchronizationId || state && state.lastSynchronizationId || state && state.lastDisconnectedSynchronizationId));
            }
            let timeLeftInSeconds = Math.max(0, timeoutInSeconds - (Date.now() - startTime) / 1000);
            const region = _this.getRegion(state.instanceIndex);
            const accountId = _this._account.accountRegions[region];
            yield _this._websocketClient.waitSynchronized(accountId, _this.getInstanceNumber(instanceIndex), applicationPattern, timeLeftInSeconds);
        })();
    }
    /**
   * Closes the connection. The instance of the class should no longer be used after this method is invoked.
   * @param {string} instanceId connection instance id
   */ close(instanceId) {
        var _this = this;
        return _async_to_generator(function*() {
            if (_this._opened) {
                _this._openedInstances = _this._openedInstances.filter((id)=>id !== instanceId);
                if (!_this._openedInstances.length && !_this._closed) {
                    clearInterval(_this._refreshJob);
                    _this._logger.debug(`${_this._account.id}: Closing connection`);
                    Object.values(_this._stateByInstanceIndex).forEach((state)=>clearTimeout(state.synchronizationTimeout));
                    _this._stateByInstanceIndex = {};
                    yield _this._connectionRegistry.removeStreaming(_this._account);
                    _this._terminalState.close();
                    const accountRegions = _this._account.accountRegions;
                    _this._websocketClient.removeSynchronizationListener(_this._account.id, _this);
                    _this._websocketClient.removeSynchronizationListener(_this._account.id, _this._terminalState);
                    _this._websocketClient.removeSynchronizationListener(_this._account.id, _this._historyStorage);
                    _this._websocketClient.removeSynchronizationListener(_this._account.id, _this._healthMonitor);
                    _this._websocketClient.removeReconnectListener(_this);
                    _this._healthMonitor.stop();
                    _this._refreshMarketDataSubscriptionSessions = {};
                    Object.values(_this._refreshMarketDataSubscriptionTimeouts).forEach((timeout)=>clearTimeout(timeout));
                    _this._refreshMarketDataSubscriptionTimeouts = {};
                    Object.values(accountRegions).forEach((replicaId)=>_this._websocketClient.removeAccountCache(replicaId));
                    _this._closed = true;
                    _this._logger.trace(`${_this._account.id}: Closed connection`);
                }
            }
        })();
    }
    /**
   * Returns synchronization status
   * @return {boolean} synchronization status
   */ get synchronized() {
        return Object.values(this._stateByInstanceIndex).reduce((acc, s)=>acc || s.synchronized, false);
    }
    /**
   * Returns MetaApi account
   * @return {MetatraderAccount} MetaApi account
   */ get account() {
        return this._account;
    }
    /**
   * Returns connection health monitor instance
   * @return {ConnectionHealthMonitor} connection health monitor instance
   */ get healthMonitor() {
        return this._healthMonitor;
    }
    _refreshMarketDataSubscriptions(accountId, instanceNumber, session) {
        var _this = this;
        return _async_to_generator(function*() {
            const region = _this._websocketClient.getAccountRegion(accountId);
            const instance = `${region}:${instanceNumber}`;
            try {
                if (_this._refreshMarketDataSubscriptionSessions[instance] === session) {
                    const subscriptionsList = [];
                    Object.keys(_this._subscriptions).forEach((key)=>{
                        const subscriptions = _this.subscriptions(key);
                        const subscriptionsItem = {
                            symbol: key
                        };
                        if (subscriptions) {
                            subscriptionsItem.subscriptions = subscriptions;
                        }
                        subscriptionsList.push(subscriptionsItem);
                    });
                    yield _this._websocketClient.refreshMarketDataSubscriptions(accountId, instanceNumber, subscriptionsList);
                }
            } catch (err) {
                _this._logger.error(`Error refreshing market data subscriptions job for account ${_this._account.id} ` + `${instanceNumber}`, err);
            } finally{
                if (_this._refreshMarketDataSubscriptionSessions[instance] === session) {
                    let refreshInterval = (Math.random() * (_this._maxSubscriptionRefreshInterval - _this._minSubscriptionRefreshInterval) + _this._minSubscriptionRefreshInterval) * 1000;
                    _this._refreshMarketDataSubscriptionTimeouts[instance] = setTimeout(()=>_this._refreshMarketDataSubscriptions(accountId, instanceNumber, session), refreshInterval);
                }
            }
        })();
    }
    _generateStopOptions(stopLoss, takeProfit) {
        let trade = {};
        if (typeof stopLoss === 'number') {
            trade.stopLoss = stopLoss;
        } else if (stopLoss) {
            trade.stopLoss = stopLoss.value;
            trade.stopLossUnits = stopLoss.units;
        }
        if (typeof takeProfit === 'number') {
            trade.takeProfit = takeProfit;
        } else if (takeProfit) {
            trade.takeProfit = takeProfit.value;
            trade.takeProfitUnits = takeProfit.units;
        }
        return trade;
    }
    _ensureSynchronized(instanceIndex, key) {
        var _this = this;
        return _async_to_generator(function*() {
            let state = _this._getState(instanceIndex);
            if (state && state.shouldSynchronize && !_this._closed) {
                try {
                    const synchronizationResult = yield _this.synchronize(instanceIndex);
                    if (synchronizationResult) {
                        state.synchronized = true;
                        state.synchronizationRetryIntervalInSeconds = 1;
                        delete state.ensureSynchronizeTimeout;
                    }
                    _this._scheduleSynchronizationTimeout(instanceIndex);
                } catch (err) {
                    const level = _this._latencyService.getSynchronizedAccountInstances(_this._account.id).length ? 'debug' : 'error';
                    _this._logger[level]('MetaApi websocket client for account ' + _this._account.id + ':' + instanceIndex + ' failed to synchronize', err);
                    if (state.shouldSynchronize === key) {
                        clearTimeout(state.ensureSynchronizeTimeout);
                        state.ensureSynchronizeTimeout = setTimeout(_this._ensureSynchronized.bind(_this, instanceIndex, key), state.synchronizationRetryIntervalInSeconds * 1000);
                        state.synchronizationRetryIntervalInSeconds = Math.min(state.synchronizationRetryIntervalInSeconds * 2, 300);
                    }
                }
            }
        })();
    }
    _getState(instanceIndex) {
        if (!this._stateByInstanceIndex['' + instanceIndex]) {
            this._stateByInstanceIndex['' + instanceIndex] = {
                instanceIndex,
                ordersSynchronized: {},
                dealsSynchronized: {},
                shouldSynchronize: undefined,
                synchronizationRetryIntervalInSeconds: 1,
                synchronized: false,
                lastDisconnectedSynchronizationId: undefined,
                lastSynchronizationId: undefined,
                disconnected: false
            };
        }
        return this._stateByInstanceIndex['' + instanceIndex];
    }
    _scheduleSynchronizationTimeout(instanceIndex) {
        let state = this._getState(instanceIndex);
        if (state && !this._closed) {
            clearTimeout(state.synchronizationTimeout);
            state.synchronizationTimeout = setTimeout(()=>this._checkSynchronizationTimedOut(instanceIndex), 2 * 60 * 1000);
            this._logger.debug(`${this._account.id}:${instanceIndex}: scheduled synchronization timeout`);
        }
    }
    _checkSynchronizationTimedOut(instanceIndex) {
        this._logger.debug(`${this._account.id}:${instanceIndex}: checking if synchronization timed out out`);
        let state = this._getState(instanceIndex);
        if (state && !this._closed) {
            let synchronizationId = state.lastSynchronizationId;
            let synchronized = !!state.dealsSynchronized[synchronizationId];
            if (!synchronized && synchronizationId && state.shouldSynchronize) {
                this._logger.warn(`${this._account.id}:${instanceIndex}: resynchronized since latest synchronization ` + `${synchronizationId} did not finish in time`);
                this._ensureSynchronized(instanceIndex, state.shouldSynchronize);
            }
        }
    }
    /**
   * Constructs MetaApi MetaTrader streaming Api connection
   * @param {MetaApiOpts} options metaapi options
   * @param {MetaApiWebsocketClient} websocketClient MetaApi websocket client
   * @param {TerminalHashManager} terminalHashManager terminal hash manager
   * @param {MetatraderAccount} account MetaTrader account id to connect to
   * @param {HistoryStorage} historyStorage terminal history storage. By default an instance of MemoryHistoryStorage
   * will be used.
   * @param {ConnectionRegistry} connectionRegistry metatrader account connection registry
   * @param {Date} [historyStartTime] history start sync time
   * @param {RefreshSubscriptionsOpts} [refreshSubscriptionsOpts] subscriptions refresh options
   */ constructor(options, websocketClient, terminalHashManager, account, historyStorage, connectionRegistry, historyStartTime, refreshSubscriptionsOpts){
        super(options, websocketClient, account);
        refreshSubscriptionsOpts = refreshSubscriptionsOpts || {};
        const validator = new OptionsValidator();
        this._minSubscriptionRefreshInterval = validator.validateNonZero(refreshSubscriptionsOpts.minDelayInSeconds, 1, 'refreshSubscriptionsOpts.minDelayInSeconds');
        this._maxSubscriptionRefreshInterval = validator.validateNonZero(refreshSubscriptionsOpts.maxDelayInSeconds, 600, 'refreshSubscriptionsOpts.maxDelayInSeconds');
        this._connectionRegistry = connectionRegistry;
        this._historyStartTime = historyStartTime;
        this._terminalHashManager = terminalHashManager;
        this._terminalState = new TerminalState(account, terminalHashManager, this._websocketClient);
        this._historyStorage = historyStorage || new MemoryHistoryStorage();
        this._healthMonitor = new ConnectionHealthMonitor(this);
        this._websocketClient.addSynchronizationListener(account.id, this);
        this._websocketClient.addSynchronizationListener(account.id, this._terminalState);
        this._websocketClient.addSynchronizationListener(account.id, this._historyStorage);
        this._websocketClient.addSynchronizationListener(account.id, this._healthMonitor);
        Object.values(account.accountRegions).forEach((replicaId)=>this._websocketClient.addReconnectListener(this, replicaId));
        this._subscriptions = {};
        this._stateByInstanceIndex = {};
        this._refreshMarketDataSubscriptionSessions = {};
        this._refreshMarketDataSubscriptionTimeouts = {};
        this._openedInstances = [];
        this._logger = LoggerManager.getLogger('MetaApiConnection');
    }
};
/**
 * Exposes MetaApi MetaTrader streaming API connection to consumers
 */ export { StreamingMetaApiConnection as default };

//# sourceMappingURL=data:application/json;base64,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