/**
 * metaapi.cloud provisioning profile API client (see https://metaapi.cloud/docs/provisioning/)
 */
 

import { Buffer } from 'buffer';
import { Version } from './metatraderAccount.client';
export class ProvisioningProfileClient {

  /**
   * Retrieves provisioning profiles owned by user
   * (see https://metaapi.cloud/docs/provisioning/api/provisioningProfile/readProvisioningProfiles/)
   * Method is accessible only with API access token
   * @param {ProvisioningProfilesFilter} [filter] provisioning profiles filter
   * @param {GetProvisioningProfilesApiVersion} [apiVersion] api version to use
   * @return {Promise<Array<ProvisioningProfileDto> | ProvisioningProfilesListDto>} promise resolving with provisioning profiles found
   */
  getProvisioningProfiles(filter?: ProvisioningProfilesFilter, apiVersion?: GetProvisioningProfilesApiVersion): Promise<Array<ProvisioningProfileDto> | ProvisioningProfilesListDto>;

  /**
   * Retrieves a provisioning profile by id (see
   * https://metaapi.cloud/docs/provisioning/api/provisioningProfile/readProvisioningProfile/). Throws an error if
   * profile is not found.
   * Method is accessible only with API access token
   * @param {string} id provisioning profile id
   * @return {Promise<ProvisioningProfileDto>} promise resolving with provisioning profile found
   */
  getProvisioningProfile(id: string): Promise<ProvisioningProfileDto>

  /**
   * Creates a new provisioning profile (see
   * https://metaapi.cloud/docs/provisioning/api/provisioningProfile/createNewProvisioningProfile/). After creating a
   * provisioning profile you are required to upload extra files in order to activate the profile for further use.
   * Method is accessible only with API access token
   * @param {NewProvisioningProfileDto} provisioningProfile provisioning profile to create
   * @return {Promise<ProvisioningProfileIdDto>} promise resolving with an id of the provisioning profile created
   */
  createProvisioningProfile(provisioningProfile: NewProvisioningProfileDto): Promise<ProvisioningProfileIdDto>;
  
  /**
   * Uploads a file to a provisioning profile (see
   * https://metaapi.cloud/docs/provisioning/api/provisioningProfile/uploadFilesToProvisioningProfile/). Uploading a
   * file by name is allowed only for Node.js.
   * Method is accessible only with API access token
   * @param {string} provisioningProfileId provisioning profile id to upload file to
   * @param {string} fileName name of the file to upload. Allowed values are servers.dat for MT5 profile, broker.srv for
   * MT4 profile
   * @param {string|Buffer} file path to a file to upload or buffer containing file contents
   * @return {Promise} promise resolving when file upload is completed
   */
  uploadProvisioningProfileFile(provisioningProfileId: string, fileName: string, file: string|Buffer): Promise<any>;

  /**
   * Deletes a provisioning profile (see
   * https://metaapi.cloud/docs/provisioning/api/provisioningProfile/deleteProvisioningProfile/). Please note that in
   * order to delete a provisioning profile you need to delete MT accounts connected to it first.
   * Method is accessible only with API access token
   * @param {string} id provisioning profile id
   * @return {Promise} promise resolving when provisioning profile is deleted
   */
  deleteProvisioningProfile(id: string): Promise<any>;

  /**
   * Updates existing provisioning profile data (see
   * https://metaapi.cloud/docs/provisioning/api/provisioningProfile/updateProvisioningProfile/).
   * Method is accessible only with API access token
   * @param {string} id provisioning profile id
   * @param {ProvisioningProfileUpdateDto} provisioningProfile updated provisioning profile
   * @return {Promise} promise resolving when provisioning profile is updated
   */
  updateProvisioningProfile(id: string, provisioningProfile: ProvisioningProfileUpdateDto): Promise<any>;
}

/**
 * Provisioning profile type
 */
export declare type ProvisioningProfileType = 'mtTerminal' | 'managerApi'

/**
 * Provisioning profile status
 */
export declare type ProvisioningProfileStatus = 'new' | 'active'

/**
 * Get provisioning profiles API version
 */
export declare type GetProvisioningProfilesApiVersion = '1' | '2'

/**
 * Provisioning profiles filter
 */
export declare type ProvisioningProfilesFilter = {

  /**
   * search offset (defaults to 0) (must be greater or equal to 0)
   */
  offset?: number,

  /**
   * search limit (defaults to 1000) 
   * (must be greater or equal to 1 and less or equal to 1000)
   */
  limit?: number,

  /**
   * MT version
   */
  version?: Version,

  /**
   * profile type
   */
  type?: ProvisioningProfileType,

  /**
   * profile status
   */
  status?: ProvisioningProfileStatus,

  /**
   * partially search over provisioning profile name to match query
   */
  query?: string,
}

/**
 * Provisioning profile model
 */
export declare type ProvisioningProfileDto = {

  /**
   * provisioning profile unique identifier
   */
  _id: string,

  /**
   * provisioning profile name
   */
  name: string,

  /**
   * MetaTrader version (allowed values are 4 and 5)
   */
  version: number,

  /**
   * provisioning profile status (allowed values are new and active)
   */
  status: string,

  /**
   * broker timezone name from Time Zone Database
   */
  brokerTimezone: string,

  /**
   * broker DST switch timezone name from Time Zone Database
   */
  brokerDSTSwitchTimezone: string
}


/**
 * Provisioning profiles list model
 */
export declare type ProvisioningProfilesListDto = {

  /**
   * provisioning profiles count
   */
  count: number,

  /**
   * provisioning profiles list
   */
  items: Array<ProvisioningProfileDto>,
}

/**
 * New provisioning profile model
 */
export declare type NewProvisioningProfileDto = {

  /**
   * provisioning profile name
   */
  name: string,

  /**
   * MetaTrader version (allowed values are 4 and 5)
   */
  version: number,

  /**
   * broker timezone name from Time Zone Database
   */
  brokerTimezone: string,

  /**
   * broker DST switch timezone name from Time Zone Database
   */
  brokerDSTSwitchTimezone: string
}

/**
 * Updated provisioning profile data
 */
export declare type ProvisioningProfileUpdateDto = {

  /**
   * provisioning profile name
   */
  name: string
}

/**
 * Provisioning profile id model
 */
export declare type ProvisioningProfileIdDto = {

  /**
   * provisioning profile unique identifier
   */
  id: string
}