"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return TrackerEventListenerManager;
    }
});
const _randomstring = /*#__PURE__*/ _interop_require_default(require("randomstring"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let TrackerEventListenerManager = class TrackerEventListenerManager {
    /**
   * Returns the dictionary of tracker event listeners
   * @returns {{[listenerId: string]: TrackerEventListener}} dictionary of tracker event listeners
   */ get trackerEventListeners() {
        return this._trackerEventListeners;
    }
    /**
   * Adds a tracker event listener
   * @param {TrackerEventListener} listener tracker event listener 
   * @param {String} [accountId] account id
   * @param {String} [trackerId] tracker id
   * @param {Number} [sequenceNumber] event sequence number
   * @returns {String} tracker event listener id
   */ addTrackerEventListener(listener, accountId, trackerId, sequenceNumber) {
        const listenerId = _randomstring.default.generate(10);
        this._trackerEventListeners[listenerId] = listener;
        this._startTrackerEventJob(listenerId, listener, accountId, trackerId, sequenceNumber);
        return listenerId;
    }
    /**
   * Removes tracker event listener by id
   * @param {String} listenerId listener id 
   */ removeTrackerEventListener(listenerId) {
        delete this._trackerEventListeners[listenerId];
    }
    async _startTrackerEventJob(listenerId, listener, accountId, trackerId, sequenceNumber) {
        let throttleTime = this._errorThrottleTime;
        while(this._trackerEventListeners[listenerId]){
            try {
                const packets = await this._domainClient.requestApi({
                    url: "/users/current/tracker-events/stream",
                    method: "GET",
                    headers: {
                        "auth-token": this._domainClient.token,
                        "api-version": "1"
                    },
                    params: {
                        previousSequenceNumber: sequenceNumber,
                        accountId,
                        trackerId,
                        limit: 1000
                    }
                }, true);
                for (let packet of packets){
                    await listener.onTrackerEvent(packet);
                }
                throttleTime = this._errorThrottleTime;
                if (this._trackerEventListeners[listenerId] && packets.length) {
                    sequenceNumber = packets.slice(-1)[0].sequenceNumber;
                }
            } catch (err) {
                listener.onError(err);
                await new Promise((res)=>setTimeout(res, throttleTime));
                throttleTime = Math.min(throttleTime * 2, 30000);
            }
        }
    }
    /**
   * Constructs tracker event listener manager instance
   * @param {DomainClient} domainClient domain client
   */ constructor(domainClient){
        this._domainClient = domainClient;
        this._trackerEventListeners = {};
        this._errorThrottleTime = 1000;
    }
};

//# sourceMappingURL=data:application/json;base64,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