"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, /**
 * Connection URL and request managing client
 */ "default", {
    enumerable: true,
    get: function() {
        return DomainClient;
    }
});
const _errorHandler = require("./errorHandler");
let DomainClient = class DomainClient {
    /**
   * Constructs domain client instance
   * @param {HttpClient} httpClient HTTP client
   * @param {String} token authorization token
   * @param {String} domain domain to connect to, default is agiliumtrade.agiliumtrade.ai
   */ constructor(httpClient, token, domain = "agiliumtrade.agiliumtrade.ai"){
        this._httpClient = httpClient;
        this._domain = domain;
        this._token = token;
        this._urlCache = null;
        this._accountCache = {};
    }
    /**
   * Returns domain client token
   * @returns {String} client token
   */ get token() {
        return this._token;
    }
    /**
   * Sends a MetaStats API request
   * @param {Function} getOpts function to get request options
   * @param {String} accountId account id
   * @returns {Object|String|any} request result
   */ async requestMetastats(getOpts, accountId) {
        await this._updateHost();
        await this._updateAccountHost(accountId);
        const accountCache = this._accountCache[accountId];
        try {
            const regionSettings = accountCache.regions[accountCache.regionIndex];
            const opts = getOpts(`https://metastats-api-v1.${regionSettings.region}.${this._urlCache.domain}`, regionSettings.id);
            return await this._httpClient.request(opts);
        } catch (err) {
            if (![
                "ConflictError",
                "InternalError",
                "ApiError",
                "TimeoutError"
            ].includes(err.name)) {
                throw err;
            } else {
                if (accountCache.regions.length === accountCache.regionIndex + 1) {
                    accountCache.regionIndex = 0;
                    throw err;
                } else {
                    accountCache.regionIndex++;
                    return await this.requestMetastats(getOpts, accountId);
                }
            }
        }
    }
    async _updateHost() {
        if (!this._urlCache || this._urlCache.lastUpdated < Date.now() - 1000 * 60 * 10) {
            const urlSettings = await this._httpClient.requestWithFailover({
                url: `https://mt-provisioning-api-v1.${this._domain}/users/current/servers/mt-client-api`,
                method: "GET",
                headers: {
                    "auth-token": this._token
                },
                json: true
            });
            this._urlCache = {
                domain: urlSettings.domain,
                lastUpdated: Date.now()
            };
        }
    }
    async _updateAccountHost(accountId) {
        if (!this._accountCache[accountId] || this._accountCache[accountId].lastUpdated < Date.now() - 1000 * 60 * 10) {
            const getAccount = async (id)=>{
                const accountOpts = {
                    url: `https://mt-provisioning-api-v1.${this._domain}/users/current/accounts/${id}`,
                    method: "GET",
                    headers: {
                        "auth-token": this._token
                    },
                    json: true
                };
                return await this._httpClient.requestWithFailover(accountOpts);
            };
            let accounts = [];
            let accountData = await getAccount(accountId);
            if (accountData.primaryAccountId) {
                accountData = await getAccount(accountData.primaryAccountId);
            }
            accounts = [
                {
                    _id: accountData._id,
                    region: accountData.region,
                    state: accountData.state
                }
            ].concat(accountData.accountReplicas || []);
            accounts = accounts.filter((account)=>account.state === "DEPLOYED");
            if (!accounts.length) {
                throw new _errorHandler.ValidationError("There are no replicas deployed yet. Please make sure at least " + "one of the replicas is deployed.");
            }
            let regions = accounts.map((account)=>({
                    region: account.region,
                    id: account._id
                }));
            this._accountCache[accountId] = {
                regions,
                regionIndex: 0,
                lastUpdated: Date.now()
            };
        }
    }
};

//# sourceMappingURL=data:application/json;base64,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