import DomainClient from './domain.client';
import { Metrics, OpenTrade, Trade } from './metaStats.client.schemas';
export * from './metaStats.client.schemas';
/**
 * metaapi.cloud MetaStats MetaTrader API client
 */
export default class MetaStatsClient {
    private _domainClient;
    /**
     * Constructs MetaStats API client instance
     * @param {DomainClient} domainClient domain client
     */
    constructor(domainClient: DomainClient);
    /**
     * Returns metrics of MetaApi account. This API call is billable
     * https://metaapi.cloud/docs/metastats/restApi/api/calculateMetrics/
     * @param {String} accountId MetaApi account id
     * @param {Boolean} [includeOpenPositions] indicates whether open positions will be included
     * in the metrics, default false
     * @return {Metrics} account metrics
     */
    getMetrics(accountId: string, includeOpenPositions?: boolean): Promise<Metrics>;
    /**
     * Returns historical trades of MetaApi account
     * https://metaapi.cloud/docs/metastats/restApi/api/getHistoricalTrades/
     * @param {String} accountId MetaApi account id
     * @param {String} startTime start of time range, inclusive
     * @param {String} endTime end of time range, exclusive
     * @param {Boolean} [updateHistory] update historical trades before returning results.
     * If set to true, the API call will be counted towards billable MetaStats API calls.
     * If set to false, the API call is not billable. Default is true
     * @param {Number} [limit] pagination limit
     * @param {Number} [offset] pagination offset
     * @param {Number} [marketValue ] trade market value
     * @return {Array<Trade>} account historical trades
     */
    getAccountTrades(accountId: string, startTime: string, endTime: string, updateHistory?: boolean, limit?: number, offset?: number): Promise<Array<Trade>>;
    /**
     * Returns open trades of MetaApi account. This API call is not billable
     * https://metaapi.cloud/docs/metastats/restApi/api/getOpenTrades/
     * @param {String} accountId MetaApi account id
     * @param {Number} [marketValue] trade market value
     * @return {Array<OpenTrade>} account historical trades
     */
    getAccountOpenTrades(accountId: string): Promise<Array<OpenTrade>>;
    /**
     * Resets metrics and trade history for a trading account from MetaStats. The data will be downloaded from trading
     * account again when you calculate the MetaStats metrics next time. This API call is not billable
     * @param accountId MetaApi account id
     * @return promise resolving when removed
     */
    resetMetrics(accountId: string): Promise<void>;
}
