"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    /**
 * HTTP client library based on request-promise
 */ default: function() {
        return HttpClient;
    },
    HttpClientMock: function() {
        return HttpClientMock;
    }
});
const _axios = /*#__PURE__*/ _interop_require_default(require("axios"));
const _errorHandler = require("./errorHandler");
const _timeoutError = /*#__PURE__*/ _interop_require_default(require("./timeoutError"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let HttpClient = class HttpClient {
    /**
   * @typedef {Object} RetryOptions retry options
   * @property {Number} [retries] the number of attempts to retry failed request, default 5
   * @property {Number} [minDelayInSeconds] minimum delay in seconds before retrying, default 1
   * @property {Number} [maxDelayInSeconds] maximum delay in seconds before retrying, default 30
   */ /**
   * Constructs HttpClient class instance
   * @param {Number} [timeout] request timeout in seconds
   * @param {Number} [extendedTimeout] request timeout in seconds
   * @param {RetryOptions} [retryOpts] retry options
   */ constructor(timeout = 10, extendedTimeout = 70, retryOpts = {}){
        this._timeout = timeout * 1000;
        this._extendedTimeout = extendedTimeout * 1000;
        this._retries = retryOpts.retries || 5;
        this._minRetryDelay = (retryOpts.minDelayInSeconds || 1) * 1000;
        this._maxRetryDelay = (retryOpts.maxDelayInSeconds || 30) * 1000;
    }
    /**
   * Performs a request. Response errors are returned as ApiError or subclasses.
   * @param {Object} options request options
   * @param {Boolean} isExtendedTimeout whether to run the request with an extended timeout
   * @returns {Object|String|any} request result
   */ async request(options, isExtendedTimeout, endTime = Date.now() + this._maxRetryDelay * this._retries) {
        options.timeout = isExtendedTimeout ? this._extendedTimeout : this._timeout;
        try {
            const response = await this._makeRequest(options);
            return response && response.data || undefined;
        } catch (err) {
            const error = this._convertError(err);
            if (error.name === "TooManyRequestsError") {
                const retryTime = Date.parse(error.metadata.recommendedRetryTime);
                const date = Date.now();
                if (retryTime < endTime) {
                    if (retryTime > date) {
                        await this._wait(retryTime - date);
                    }
                    return await this.request(options, isExtendedTimeout, endTime);
                } else {
                    throw error;
                }
            } else {
                throw error;
            }
        }
    }
    /**
   * Performs a request with a failover. Response errors are returned as ApiError or subclasses.
   * @param {Object} options request options
   * @returns {Object|String|any} request result
   */ async requestWithFailover(options, retryCounter = 0, endTime = Date.now() + this._maxRetryDelay * this._retries) {
        options.timeout = this._timeout;
        let retryAfterSeconds = 0;
        options.callback = (e, res)=>{
            if (res && res.status === 202) {
                retryAfterSeconds = res.headers["retry-after"];
            }
        };
        let body;
        try {
            const response = await this._makeRequest(options);
            options.callback(null, response);
            body = response && response.data || undefined;
        } catch (err) {
            retryCounter = await this._handleError(err, retryCounter, endTime);
            return this.requestWithFailover(options, retryCounter, endTime);
        }
        if (retryAfterSeconds) {
            await this._handleRetry(endTime, retryAfterSeconds * 1000);
            body = await this.requestWithFailover(options, retryCounter, endTime);
        }
        return body;
    }
    _makeRequest(options) {
        return (0, _axios.default)({
            transitional: {
                clarifyTimeoutError: true
            },
            ...options
        });
    }
    async _wait(pause) {
        await new Promise((res)=>setTimeout(res, pause));
    }
    async _handleRetry(endTime, retryAfter) {
        if (endTime > Date.now() + retryAfter) {
            await this._wait(retryAfter);
        } else {
            throw new _timeoutError.default("Timed out waiting for the response");
        }
    }
    async _handleError(err, retryCounter, endTime) {
        const error = this._convertError(err);
        if ([
            "ConflictError",
            "InternalError",
            "ApiError",
            "TimeoutError"
        ].includes(error.name) && retryCounter < this._retries) {
            const pause = Math.min(Math.pow(2, retryCounter) * this._minRetryDelay, this._maxRetryDelay);
            await this._wait(pause);
            return retryCounter + 1;
        } else if (error.name === "TooManyRequestsError") {
            const retryTime = Date.parse(error.metadata.recommendedRetryTime);
            if (retryTime < endTime) {
                await this._wait(retryTime - Date.now());
                return retryCounter;
            }
        }
        throw error;
    }
    // eslint-disable-next-line complexity
    _convertError(err) {
        const errorResponse = err.response || {};
        const errorData = errorResponse.data || {};
        const status = errorResponse.status || err.status;
        const url = err?.config?.url;
        const errMsg = errorData.message || err.message;
        const errMsgDefault = errorData.message || err.code || err.message;
        switch(status){
            case 400:
                return new _errorHandler.ValidationError(errMsg, errorData.details || err.details, url);
            case 401:
                return new _errorHandler.UnauthorizedError(errMsg, url);
            case 403:
                return new _errorHandler.ForbiddenError(errMsg, url);
            case 404:
                return new _errorHandler.NotFoundError(errMsg, url);
            case 409:
                return new _errorHandler.NotFoundError(errMsg, url);
            case 429:
                return new _errorHandler.TooManyRequestsError(errMsg, errorData.metadata || err.metadata, url);
            case 500:
                return new _errorHandler.InternalError(errMsg, url);
            default:
                return new _errorHandler.ApiError(_errorHandler.ApiError, errMsgDefault, status, url);
        }
    }
};
let HttpClientMock = class HttpClientMock extends HttpClient {
    /**
   * Constructs HTTP client mock
   * @param {Function(options:Object):Promise} requestFn mocked request function
   * @param {Number} timeout request timeout in seconds
   * @param {RetryOptions} retryOpts retry options
   */ constructor(requestFn, timeout, extendedTimeout, retryOpts){
        super(timeout, extendedTimeout, retryOpts);
        this._requestFn = requestFn;
    }
    _makeRequest() {
        return this._requestFn.apply(this, arguments);
    }
};

//# sourceMappingURL=data:application/json;base64,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