'use strict';
function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
import MetaApiClient from '../../metaapi.client';
import randomstring from 'randomstring';
import LoggerManager from '../../../logger';
let UserLogListenerManager = class UserLogListenerManager extends MetaApiClient {
    /**
   * Returns the dictionary of strategy log listeners
   * @returns {Object} dictionary of strategy log listeners
   */ get strategyLogListeners() {
        return this._strategyLogListeners;
    }
    /**
   * Returns the dictionary of subscriber log listeners
   * @returns {Object} dictionary of subscriber log listeners
   */ get subscriberLogListeners() {
        return this._subscriberLogListeners;
    }
    /**
   * Adds a strategy log listener
   * @param {UserLogListener} listener user log listener
   * @param {String} strategyId strategy id
   * @param {Date} [startTime] log search start time
   * @param {String} [positionId] position id filter
   * @param {'DEBUG'|'INFO'|'WARN'|'ERROR'} [level] minimum severity level
   * @param {Number} [limit] log pagination limit
   * @returns {String} strategy log listener id
   */ addStrategyLogListener(listener, strategyId, startTime, positionId, level, limit) {
        const listenerId = randomstring.generate(10);
        this._strategyLogListeners[listenerId] = listener;
        this._startStrategyLogStreamJob(listenerId, listener, strategyId, startTime, positionId, level, limit);
        return listenerId;
    }
    /**
   * Adds a subscriber log listener
   * @param {UserLogListener} listener user log listener
   * @param {String} subscriberId subscriber id
   * @param {Date} [startTime] log search start time
   * @param {string} [strategyId] strategy id filter
   * @param {string} [positionId] position id filter
   * @param {'DEBUG'|'INFO'|'WARN'|'ERROR'} [level] minimum severity level
   * @param {Number} [limit] log pagination limit
   * @returns {String} subscriber log listener id
   */ addSubscriberLogListener(listener, subscriberId, startTime, strategyId, positionId, level, limit) {
        const listenerId = randomstring.generate(10);
        this._subscriberLogListeners[listenerId] = listener;
        this._startSubscriberLogStreamJob(listenerId, listener, subscriberId, startTime, strategyId, positionId, level, limit);
        return listenerId;
    }
    /**
   * Removes strategy log listener by id
   * @param {String} listenerId listener id 
   */ removeStrategyLogListener(listenerId) {
        delete this._strategyLogListeners[listenerId];
    }
    /**
   * Removes subscriber log listener by id
   * @param {String} listenerId listener id 
   */ removeSubscriberLogListener(listenerId) {
        delete this._subscriberLogListeners[listenerId];
    }
    _startStrategyLogStreamJob(listenerId, listener, strategyId, startTime, positionId, level, limit) {
        var _this = this;
        return _async_to_generator(function*() {
            let throttleTime = _this._errorThrottleTime;
            while(_this._strategyLogListeners[listenerId]){
                const opts = {
                    url: `/users/current/strategies/${strategyId}/user-log/stream`,
                    method: 'GET',
                    params: {
                        startTime,
                        positionId,
                        level,
                        limit
                    },
                    headers: {
                        'auth-token': _this._token
                    },
                    json: true
                };
                try {
                    const packets = yield _this._domainClient.requestCopyFactory(opts, true);
                    // stop job if user has unsubscribed in time of new packets has been received
                    if (!_this._strategyLogListeners[listenerId]) {
                        return;
                    }
                    yield listener.onUserLog(packets);
                    throttleTime = _this._errorThrottleTime;
                    if (_this._strategyLogListeners[listenerId] && packets.length) {
                        startTime = new Date(new Date(packets[0].time).getTime() + 1);
                    }
                } catch (err) {
                    yield listener.onError(err);
                    if (err.name === 'NotFoundError') {
                        _this._logger.error(`Strategy ${strategyId} not found, removing listener ${listenerId}`);
                        delete _this._strategyLogListeners[listenerId];
                    } else {
                        _this._logger.error(`Failed to retrieve user log stream for strategy ${strategyId}, ` + `listener ${listenerId}, retrying in ${Math.floor(throttleTime / 1000)} seconds`, err);
                        yield new Promise((res)=>setTimeout(res, throttleTime));
                        throttleTime = Math.min(throttleTime * 2, 30000);
                    }
                }
            }
        })();
    }
    _startSubscriberLogStreamJob(listenerId, listener, subscriberId, startTime, strategyId, positionId, level, limit) {
        var _this = this;
        return _async_to_generator(function*() {
            let throttleTime = _this._errorThrottleTime;
            while(_this._subscriberLogListeners[listenerId]){
                const opts = {
                    url: `/users/current/subscribers/${subscriberId}/user-log/stream`,
                    method: 'GET',
                    params: {
                        startTime,
                        strategyId,
                        positionId,
                        level,
                        limit
                    },
                    headers: {
                        'auth-token': _this._token
                    },
                    json: true
                };
                try {
                    const packets = yield _this._domainClient.requestCopyFactory(opts, true);
                    // stop job if user has unsubscribed in time of new packets has been received
                    if (!_this._subscriberLogListeners[listenerId]) {
                        return;
                    }
                    yield listener.onUserLog(packets);
                    throttleTime = _this._errorThrottleTime;
                    if (_this._subscriberLogListeners[listenerId] && packets.length) {
                        startTime = new Date(new Date(packets[0].time).getTime() + 1);
                    }
                } catch (err) {
                    yield listener.onError(err);
                    if (err.name === 'NotFoundError') {
                        _this._logger.error(`Subscriber ${subscriberId} not found, removing listener ${listenerId}`);
                        delete _this._subscriberLogListeners[listenerId];
                    } else {
                        _this._logger.error(`Failed to retrieve user log stream for subscriber ${subscriberId}, ` + `listener ${listenerId}, retrying in ${Math.floor(throttleTime / 1000)} seconds`, err);
                        yield new Promise((res)=>setTimeout(res, throttleTime));
                        throttleTime = Math.min(throttleTime * 2, 30000);
                    }
                }
            }
        })();
    }
    /**
   * Constructs user log event listener manager instance
   * @param {DomainClient} domainClient domain client
   */ constructor(domainClient){
        super(domainClient);
        this._domainClient = domainClient;
        this._strategyLogListeners = {};
        this._subscriberLogListeners = {};
        this._errorThrottleTime = 1000;
        this._logger = LoggerManager.getLogger('UserLogListenerManager');
    }
};
/**
 * User log event listener manager
 */ export { UserLogListenerManager as default };

//# sourceMappingURL=data:application/json;base64,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