import DomainClient from '../domain.client';
import MetaApiClient from '../metaapi.client';
import { ClassicPaginationList, CopyFactoryCloseInstructions, CopyFactoryPortfolioStrategy, CopyFactoryPortfolioStrategyUpdate, CopyFactoryStrategy, CopyFactoryStrategyUpdate, CopyFactorySubscriber, CopyFactorySubscriberUpdate, Webhook, GetPortfolioStrategiesOptions, GetStrategiesOptions, GetSubscribersOptions, GetWebhooksOptions, StrategyId, NewWebhook, WebhookIdAndUrl, WebhookUpdate } from './configuration.client.schemas';
export * from './configuration.client.schemas';
/**
 * metaapi.cloud CopyFactory configuration API (trade copying configuration API) client (see
 * https://metaapi.cloud/docs/copyfactory/)
 */
export default class ConfigurationClient extends MetaApiClient {
    /**
     * Constructs CopyFactory configuration API client instance
     * @param {DomainClient} domainClient domain client
     */
    constructor(domainClient: DomainClient);
    /**
     * Strategy id
     * @typedef {Object} StrategyId
     * @property {String} id strategy id
     */
    /**
     * Retrieves new unused strategy id. Method is accessible only with API access token. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/generateStrategyId/
     * @return {Promise<StrategyId>} promise resolving with strategy id generated
     */
    generateStrategyId(): Promise<StrategyId>;
    /**
     * Generates random account id
     * @return {String} account id
     */
    generateAccountId(): string;
    /**
     * Retrieves CopyFactory copy trading strategies with pagination in infinite scroll style. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/getStrategies/
     * @param {GetStrategiesOptions} [options] options
     * @return {Promise<Array<CopyFactoryStrategy>>} promise resolving with CopyFactory strategies found
     */
    getStrategiesWithInfiniteScrollPagination(options?: GetStrategiesOptions): Promise<Array<CopyFactoryStrategy>>;
    /**
     * Retrieves CopyFactory copy trading strategies with pagination in classic style. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/getStrategies/
     * @param {GetStrategiesOptions} [options] options
     * @return {Promise<ClassicPaginationList<CopyFactoryStrategy>>} promise resolving with CopyFactory strategies found
     */
    getStrategiesWithClassicPagination(options?: GetStrategiesOptions): Promise<ClassicPaginationList<CopyFactoryStrategy>>;
    private _getStrategies;
    /**
     * Retrieves CopyFactory copy trading strategy by id. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/getStrategy/
     * @param {string} strategyId trading strategy id
     * @return {Promise<CopyFactoryStrategy>} promise resolving with CopyFactory strategy found
     */
    getStrategy(strategyId: string): Promise<CopyFactoryStrategy>;
    /**
     * Updates a CopyFactory strategy. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/updateStrategy/
     * @param {String} strategyId copy trading strategy id
     * @param {CopyFactoryStrategyUpdate} strategy trading strategy update
     * @return {Promise} promise resolving when strategy is updated
     */
    updateStrategy(strategyId: string, strategy: CopyFactoryStrategyUpdate): Promise<void>;
    /**
     * Deletes a CopyFactory strategy. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/removeStrategy/
     * @param {String} strategyId copy trading strategy id
     * @param {CopyFactoryCloseInstructions} [closeInstructions] strategy close instructions
     * @return {Promise} promise resolving when strategy is removed
     */
    removeStrategy(strategyId: string, closeInstructions?: CopyFactoryCloseInstructions): Promise<void>;
    /**
     * Retrieves CopyFactory copy portfolio strategies with pagination in infinite scroll style. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/getPortfolioStrategies/
     * @param {GetPortfolioStrategiesOptions} [options] options
     * @return {Promise<Array<CopyFactoryPortfolioStrategy>>} promise resolving with CopyFactory portfolio strategies
     */
    getPortfolioStrategiesWithInfiniteScrollPagination(options?: GetPortfolioStrategiesOptions): Promise<Array<CopyFactoryPortfolioStrategy>>;
    /**
     * Retrieves CopyFactory copy portfolio strategies with pagination in classic style. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/getPortfolioStrategies/
     * @param {GetPortfolioStrategiesOptions} [options] options
     * @return {Promise<ClassicPaginationList<CopyFactoryStrategy>>} promise resolving with CopyFactory strategies found
     */
    getPortfolioStrategiesWithClassicPagination(options?: GetPortfolioStrategiesOptions): Promise<ClassicPaginationList<CopyFactoryStrategy>>;
    private _getPortfolioStrategies;
    /**
     * Retrieves CopyFactory copy portfolio strategy by id. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/getPortfolioStrategy/
     * @param {string} portfolioId portfolio strategy id
     * @return {Promise<CopyFactoryPortfolioStrategy>} promise resolving with CopyFactory portfolio strategy found
     */
    getPortfolioStrategy(portfolioId: string): Promise<CopyFactoryPortfolioStrategy>;
    /**
     * Updates a CopyFactory portfolio strategy. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/updatePortfolioStrategy/
     * @param {String} portfolioId copy trading portfolio strategy id
     * @param {CopyFactoryPortfolioStrategyUpdate} portfolio portfolio strategy update
     * @return {Promise} promise resolving when portfolio strategy is updated
     */
    updatePortfolioStrategy(portfolioId: string, portfolio: CopyFactoryPortfolioStrategyUpdate): Promise<void>;
    /**
     * Deletes a CopyFactory portfolio strategy. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/removePortfolioStrategy/
     * @param {String} portfolioId portfolio strategy id
     * @param {CopyFactoryCloseInstructions} [closeInstructions] strategy close instructions
     * @return {Promise} promise resolving when portfolio strategy is removed
     */
    removePortfolioStrategy(portfolioId: string, closeInstructions?: CopyFactoryCloseInstructions): Promise<void>;
    /**
     * Deletes a CopyFactory portfolio strategy member. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/removePortfolioStrategyMember/
     * @param {String} portfolioId portfolio strategy id
     * @param {String} strategyId id of the strategy to delete member for
     * @param {CopyFactoryCloseInstructions} [closeInstructions] strategy close instructions
     * @return {Promise} promise resolving when portfolio strategy is removed
     */
    removePortfolioStrategyMember(portfolioId: string, strategyId: string, closeInstructions?: CopyFactoryCloseInstructions): Promise<void>;
    /**
     * Returns CopyFactory subscribers the user has configured with pagination in infinite scroll style. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/history/getSubscribers/
     * @param {GetSubscribersOptions} [options] options
     * @return {Promise<Array<CopyFactorySubscriber>>} promise resolving with subscribers found
     */
    getSubscribersWithInfiniteScrollPagination(options?: GetSubscribersOptions): Promise<Array<CopyFactorySubscriber>>;
    /**
     * Returns CopyFactory subscribers the user has configured with pagination in classic style. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/history/getSubscribers/
     * @param {GetSubscribersOptions} [options] options
     * @return {Promise<ClassicPaginationList<CopyFactorySubscriber>>} promise resolving with subscribers found
     */
    getSubscribersWithClassicPagination(options?: GetSubscribersOptions): Promise<ClassicPaginationList<CopyFactorySubscriber>>;
    private _getSubscribers;
    /**
     * Returns CopyFactory subscriber by id. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/getSubscriber/
     * @param {String} subscriberId subscriber id
     * @returns {Promise<CopyFactorySubscriber>} promise resolving with subscriber found
     */
    getSubscriber(subscriberId: string): Promise<CopyFactorySubscriber>;
    /**
     * Updates CopyFactory subscriber configuration. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/updateSubscriber/
     * @param {String} subscriberId subscriber id
     * @param {CopyFactorySubscriberUpdate} subscriber subscriber update
     * @returns {Promise} promise resolving when subscriber is updated
     */
    updateSubscriber(subscriberId: string, subscriber: CopyFactorySubscriberUpdate): Promise<void>;
    /**
     * Deletes subscriber configuration. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/removeSubscriber/
     * @param {String} subscriberId subscriber id
     * @param {CopyFactoryCloseInstructions} [closeInstructions] subscriber close instructions
     * @returns {Promise} promise resolving when subscriber is removed
     */
    removeSubscriber(subscriberId: string, closeInstructions?: CopyFactoryCloseInstructions): Promise<void>;
    /**
     * Deletes a subscription of subscriber to a strategy. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/removeSubscription/
     * @param {String} subscriberId subscriber id
     * @param {String} strategyId strategy id
     * @param {CopyFactoryCloseInstructions} [closeInstructions] subscriber close instructions
     * @returns {Promise} promise resolving when subscriber is removed
     */
    removeSubscription(subscriberId: string, strategyId: string, closeInstructions?: CopyFactoryCloseInstructions): Promise<void>;
    /**
     * Retrieves CopyFactory user webhooks list with pagination in infinite scroll style. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/getWebhooks/
     * @param strategyId strategy ID
     * @param options additional options
     * @return promise resolving with webhooks found
     */
    getWebhooksWithInfiniteScrollPagination(strategyId: string, options?: GetWebhooksOptions): Promise<Webhook[]>;
    /**
     * Retrieves CopyFactory user webhooks list with pagination in classic style. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/getWebhooks/
     * @param strategyId strategy ID
     * @param options additional options
     * @return promise resolving with webhooks found
     */
    getWebhooksWithClassicPagination(strategyId: string, options?: GetWebhooksOptions): Promise<ClassicPaginationList<Webhook>>;
    private _getWebhooks;
    /**
     * Creates a new webhook. The webhook can be used for an external app (e.g. TradingView) to submit trading signals to
     * CopyFactory. See https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/createWebhook/
     * @param strategyId strategy ID
     * @param webhook webhook
     * @returns promise resolving with created webhook ID and URL
     */
    createWebhook(strategyId: string, webhook: NewWebhook): Promise<WebhookIdAndUrl>;
    /**
     * Updates a webhook. See https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/updateWebhook/
     * @param strategyId webhook strategy ID
     * @param webhookId webhook ID
     * @param update webhook update
     * @returns promise resolving when updated
     */
    updateWebhook(strategyId: string, webhookId: string, update: WebhookUpdate): Promise<void>;
    /**
     * Deletes a webhook. See https://metaapi.cloud/docs/copyfactory/restApi/api/configuration/deleteWebhook/
     * @param strategyId webhook strategy ID
     * @param webhookId webhook ID
     * @returns promise resolving when deleted
     */
    deleteWebhook(strategyId: string, webhookId: string): Promise<void>;
}
